
#if !defined(AFX_VIMOSCONNECTION_H__ABDBA11D_4CA8_4943_B3E2_B1A7684A5078__INCLUDED_)
#define AFX_VIMOSCONNECTION_H__ABDBA11D_4CA8_4943_B3E2_B1A7684A5078__INCLUDED_

#include "errors.h"

#if _MSC_VER > 1000
#pragma once
#pragma pack(push)
#pragma pack(1)
#endif // _MSC_VER > 1000

/// Library namespace
namespace VIMOS
{

/**
 * Angle.
 * Measured in 1000th of radian.
 * In the range 0..6283
 */
typedef short UPM_ANGLE;

class CVimosPort;
class CPointList;
struct UPM_POINT;
struct MEM_POINT_ITEM;

/**
 * Connection to VIMOS system running on a camera or PC.
 *
 * This is the main class of the library.
 * It allows you to establish connection to a VIMOS instance and exchange data with it.
 * Multiple objects of this class can be used to communicate with multiple VIMOS instances
 * (e.g. on multiple cameras).
 * 
 * All communication is done synchronously. That is, the functions return when all data transmission
 * is complete. So, for best performance it is advised to run each connection in a dedicated thread.
 *
 * Objects of this class are not thread safe. This means each object should be accessed 
 * by only one thread. Nevertheless, multiple objects can be used in separate threads.
 *
 * Before any communication can take place the connection should be opened with Open().
 * When the connection is no longer necessary call Close() to release the communication
 * resources (like serial port).
 */
class CVimosConnection  
{
public:
   /// Initializes an empty object, call Open() to establish the connection.
	CVimosConnection();

   /// Automatically closes the connection if still open
   ~CVimosConnection();

public:
   /**
    * Establish connection with VIMOS.
    * Call this before any send/receive function.
    *
    * @param port_name port name ("COM1", "COM2", ...)
    * @param baud_rate baud rate (1200, 2400, 4800, 9600, 14400, 19200, 28800, 38400, 57600, 115200)
    *
    * @return IO_RESULT
    */
   IO_RESULT Open(const char *port_name, int baud_rate);

   /**
    * Close the connection.
    * After this you cannot use send/receive functions until you call Open() again.
    *
    * @return IO_RESULT
    */
   IO_RESULT Close();

   /**
    * Check if the connection is already open.
    *
    * @return bool
    */
   bool IsOpen();

   /// Data transfer format
   enum Transfer { 
      HEX = 0,      ///< Each byte is transmitted as a 2-digit hexadecimal number
      BIN,          ///< Raw binary format
      ASCII,        ///< ASCII text string
      ASCII_prompt  ///< ASCII text string preceded by a prompt string
   };

   /**
    * Send a floating-point number.
    * This value can be received by <em>Receive Result</em> tool in VIMOS.
    * This tool should be properly configured.
    * - Tool argument <em>Value type</em> should be set to \b R.
    * - Tool argument <em>Data type</em> should match function parameter \a transfer.
    * - Tool argument <em>Start byte</em> should match function parameter \a start_byte.
    * - Tool argument <em>Checksum</em> should match function parameter \a checksum.
    *
    * @param number     floating-point number to send
    * @param transfer   data transfer format (see CVimosConnection::Transfer),
    *                   should match tool argument <em>Data type</em>
    * @param start_byte start byte used for synchronization, 
    *                   should match tool argument <em>Start byte</em>
    * @param checksum   enable/disable checksum,
    *                   should match tool argument <em>Checksum</em>
    *
    * @return IO_RESULT
    */
   IO_RESULT SendResult(float number, Transfer transfer, BYTE start_byte, bool checksum);

   /**
    * Send a point.
    * This value can be received by <em>Receive Result</em> tool in VIMOS.
    * This tool should be properly configured.
    * - Tool argument <em>Value type</em> should be set to \b P.
    * - Tool argument <em>Data type</em> should match function parameter \a transfer.
    * - Tool argument <em>Start byte</em> should match function parameter \a start_byte.
    * - Tool argument <em>Checksum</em> should match function parameter \a checksum.
    *
    * @param point      point to send (see UPM_POINT)
    * @param transfer   data transfer format (see CVimosConnection::Transfer),
    *                   should match tool argument <em>Data type</em>
    * @param start_byte start byte used for synchronization, 
    *                   should match tool argument <em>Start byte</em>
    * @param checksum   enable/disable checksum,
    *                   should match tool argument <em>Checksum</em>
    *
    * @return IO_RESULT
    */
   IO_RESULT SendResult(const UPM_POINT &point, Transfer transfer, BYTE start_byte, bool checksum);

   /**
    * Send an angle.
    * This value can be received by <em>Receive Result</em> tool in VIMOS.
    * This tool should be properly configured.
    * - Tool argument <em>Value type</em> should be set to \b A.
    * - Tool argument <em>Data type</em> should match function parameter \a transfer.
    * - Tool argument <em>Start byte</em> should match function parameter \a start_byte.
    * - Tool argument <em>Checksum</em> should match function parameter \a checksum.
    *
    * @param angle      angle to send (see UPM_ANGLE)
    * @param transfer   data transfer format (see CVimosConnection::Transfer),
    *                   should match tool argument <em>Data type</em>
    * @param start_byte start byte used for synchronization, 
    *                   should match tool argument <em>Start byte</em>
    * @param checksum   enable/disable checksum,
    *                   should match tool argument <em>Checksum</em>
    *
    * @return IO_RESULT
    */
   IO_RESULT SendResult(UPM_ANGLE angle, Transfer transfer, BYTE start_byte, bool checksum);

   /**
    * Receive a floating-point number.
    * This value can be sent by <em>Send Result</em> tool in VIMOS.
    * This tool should be properly configured.
    * - Tool argument <em>Value</em> should be of type \b R.
    * - Tool argument <em>Send start byte</em> should be set to \c Yes.
    * - Tool argument <em>Send data</em> should be set to \c Yes.
    * - Tool argument <em>Data type</em> should match function parameter \a transfer.
    * - Tool argument <em>Start byte</em> should match function parameter \a start_byte.
    * - Tool argument <em>Checksum</em> should match function parameter \a checksum.
    *
    * @param number     received floating-point number
    * @param transfer   data transfer format (see CVimosConnection::Transfer),
    *                   should match tool argument <em>Data type</em>
    * @param start_byte start byte used for synchronization, 
    *                   should match tool argument <em>Start byte</em>
    * @param checksum   enable/disable checksum,
    *                   should match tool argument <em>Checksum</em>
    * @param wait_ms    time in milliseconds to wait for the data, 
    *                   \c 0 - don't wait, \c -1 - wait indefinitely
    *
    * @return IO_RESULT
    */
   IO_RESULT ReceiveResult(float &number, Transfer transfer, BYTE start_byte, bool checksum, long wait_ms);

   /**
    * Receive a point.
    * This value can be sent by <em>Send Result</em> tool in VIMOS.
    * This tool should be properly configured.
    * - Tool argument <em>Value</em> should be of type \b P.
    * - Tool argument <em>Send start byte</em> should be set to \c Yes.
    * - Tool argument <em>Send data</em> should be set to \c Yes.
    * - Tool argument <em>Data type</em> should match function parameter \a transfer.
    * - Tool argument <em>Start byte</em> should match function parameter \a start_byte.
    * - Tool argument <em>Checksum</em> should match function parameter \a checksum.
    *
    * @param point      received point (see UPM_POINT)
    * @param transfer   data transfer format (see CVimosConnection::Transfer),
    *                   should match tool argument <em>Data type</em>
    * @param start_byte start byte used for synchronization, 
    *                   should match tool argument <em>Start byte</em>
    * @param checksum   enable/disable checksum,
    *                   should match tool argument <em>Checksum</em>
    * @param wait_ms    time in milliseconds to wait for the data, 
    *                   \c 0 - don't wait, \c -1 - wait indefinitely
    *
    * @return IO_RESULT
    */
   IO_RESULT ReceiveResult(UPM_POINT &point, Transfer transfer, BYTE start_byte, bool checksum, long wait_ms);

   /**
    * Receive an angle.
    * This value can be sent by <em>Send Result</em> tool in VIMOS.
    * This tool should be properly configured.
    * - Tool argument <em>Value</em> should be of type \b A.
    * - Tool argument <em>Send start byte</em> should be set to \c Yes.
    * - Tool argument <em>Send data</em> should be set to \c Yes.
    * - Tool argument <em>Data type</em> should match function parameter \a transfer.
    * - Tool argument <em>Start byte</em> should match function parameter \a start_byte.
    * - Tool argument <em>Checksum</em> should match function parameter \a checksum.
    *
    * @param angle      received angle (see UPM_ANGLE)
    * @param transfer   data transfer format (see CVimosConnection::Transfer),
    *                   should match tool argument <em>Data type</em>
    * @param start_byte start byte used for synchronization, 
    *                   should match tool argument <em>Start byte</em>
    * @param checksum   enable/disable checksum,
    *                   should match tool argument <em>Checksum</em>
    * @param wait_ms    time in milliseconds to wait for the data, 
    *                   \c 0 - don't wait, \c -1 - wait indefinitely
    *
    * @return IO_RESULT
    */
   IO_RESULT ReceiveResult(UPM_ANGLE &angle, Transfer transfer, BYTE start_byte, bool checksum, long wait_ms);

   /**
    * Send text string.
    * The string can be received by <em>Receive String</em> tool in VIMOS.
    *
    * @param data       null-terminated text string to send
    * @param read_echo  if \c true (default), read echo returned by <em>Receive String</em> tool
    *                   and compare it with sent \a data
    *
    * @return IO_RESULT
    *         - IO_R_RECV_DATA_ERR - echo doesn't match sent \a data
    */
   IO_RESULT SendString(const char *data, bool read_echo = true);

   /**
    * Receive text string.
    * The string can be sent by <em>Send String</em> tool in VIMOS.
    *
    * @param data       buffer for received text string, will be null-terminated
    * @param data_size  size of \a data[] buffer, including space for terminating null character
    * @param send_echo  if \c false (default), don't send back received characters
    *
    * @return IO_RESULT
    */
   IO_RESULT ReceiveString(char *data, long data_size, bool send_echo = false);

   /**
    * Send point-list.
    * The point-list can be received by <em>Receive Point-list</em> tool in VIMOS.
    * This tool should be properly configured.
    * - Tool argument <em>Transfer</em> should match function parameter \a ascii.
    *
    * @param data       array of MEM_POINT_ITEM structures representing the point-list to be sent
    * @param item_count point-list length (length of \a data[] array)
    * @param ascii      should match tool argument <em>Transfer</em>, 
    *                   \c true = ASCII, \c false = Binary
    *
    * @return IO_RESULT
    */
   IO_RESULT SendPointList(MEM_POINT_ITEM *data, long item_count, bool ascii);

   /**
    * Receive point-list.
    * The point-list can be sent by <em>Send Point-list</em> tool in VIMOS.
    * This tool should be properly configured.
    * - Tool argument <em>Transfer</em> should match function parameter \a ascii.
    *
    * @param ptlist received point-list, CPointList manages the storage
    * @param ascii  should match tool argument <em>Transfer</em>,
    *               \c true = ASCII, \c false = Binary
    * @param wait   \c true - wait indefinitely, \c false - wait until the timeout expires
    *
    * @return IO_RESULT
    */
   IO_RESULT ReceivePointList(CPointList &ptlist, bool ascii, bool wait);

   /**
    * Receive image.
    * The image can be sent by <em>Send Image Area</em> tool in VIMOS.
    * Currently (as of VIMOS 2.51) this tool is not supported on PC.
    * So you can use this function get an image from the camera, but not from VIMOS Simulator.
    * The image is transmitted in JPEG format. Quality is controlled by <em>Send Image Area</em> tool.
    * If the image is received successfully, it is saved as standard JPEG file.
    *
    * @param file_name file where image will be saved, will be overwritten if already exists
    * @param wait_ms   time to wait in milliseconds for start of transmission,
    *                  \c 0 - don't wait, \c -1 - wait indefinitely
    *
    * @return IO_RESULT
    */
   IO_RESULT ReceiveImage(const char *file_name, long wait_ms);   

// implementation
protected:
   CVimosPort *pPort;

private:
   IO_RESULT SendResult(int Res_type, void *Data, int Data_Size, int Flag_bin, int stbyte,   
      int chsum, int sendb, int sendd);
   IO_RESULT ReceiveResult(int Res_Type, void *Data, int *Data_Size, int Flag_wait, short stbyte,    
      short datatype, int chsum, long wait_ms);
};

} // namespace VIMOS

#if _MSC_VER > 1000
#pragma pack(pop)
#endif // _MSC_VER > 1000

#endif // !defined(AFX_VIMOSCONNECTION_H__ABDBA11D_4CA8_4943_B3E2_B1A7684A5078__INCLUDED_)
